/*
 * Depmod
 *
 * Copyright 1994, 1995, 1996, 1997 Jacques Gelinas <jack@solucorp.qc.ca>
 * Additional modifications: Bjrn Ekwall <bj0rn@blox.se> February, March 1999
 *
 * This file is part of the Linux modutils.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <getopt.h>
#include <ctype.h>
#include <syslog.h>
#include "depmod.h"
extern "C" {
#include "module.h"
#include "version.h"
#include "util.h"
#include "config.h"
#include "obj.h"
#include "modstat.h"
}

/*
 * Add the symbol defined in the kernel, simulating a pseudo module.
 *
 * Load from the currently running kernel
 * OR
 * from a file containing kernel symbols from a different kernel.
 * The file can be either a "System.map" file or a copy of "/proc/ksyms".
 * Use a copy of "/proc/ksyms" if you are using versioned symbols!
 *
 * Return -1 if any error.
 */
static int depmod_addksyms(MODULES & mods, SYMBOLS & syms, char *file_syms)
{
	MODULE *mod = mods.setdummy("-");
	SYMBOL *tbsym[10000];
	int nbsym;
	int need;
	struct module_symbol *ksym;
	unsigned int so_far = 0;

	FILE *fp;
	int is_mapfile = 0;
	char line[100];
	char *p;

	/* Fake the _mod_use_count_ symbol provided by insmod */
	/* Dummy */
	tbsym[0] = syms.add("mod_use_count_", mod, SYM_DEFINED, need, 0);
	tbsym[1] = syms.add("__this_module", mod, SYM_DEFINED, need, 0);
	nbsym = 2;

	/*
	 * Specification: depmod / kernel syms only
	 * When initialising its symbol table from the kernel
	 * depmod silently discards all symbol from loaded modules.
	 *
	 * This means that depmod may be used at any time to compute
	 * the dependancy table, even if there are modules already
	 * loaded.
	 *
	 * depmod use the kernel system call to obtain the
	 * symbol table, not /proc/ksyms.
	 */

	if (file_syms) {
		if ((fp = fopen(file_syms, "r")) == NULL) {
			error("Can't read %s", file_syms);
			return -1;
		}
		if (!fgets(line, 100, fp)) {
			fclose(fp);
			error("premature EOF on %s", file_syms);
			return -1;
		}

		if (!strtok(line, " \t\n")) {
			fclose(fp);
			error("Illegal format of %s", file_syms);
			return -1;
		}
		p = strtok(NULL, " \t\n");
		/* The second word is either the symbol name or a type */
		if (p && strlen(p) == 1) { /* System.map */
			is_mapfile = 1;
			if (!isupper(*p))
				p = NULL;
			else
				p = strtok(NULL, " \t\n");
		} else { /* /proc/ksyms copy */
			if (p && strtok(NULL, " \t\n"))
				p = NULL;
		}

		if (p) {
			tbsym[nbsym++] = syms.add(p, mod, SYM_DEFINED, need, 0);
		}

		while (fgets(line, 100, fp)) {
			if (!is_mapfile && strchr(line, '['))
				continue;
			strtok(line, " \t\n"); /* Skip first word */
			p = strtok(NULL, " \n");
			if (is_mapfile) {
				if (!isupper(*p))
					continue;
				p = strtok(NULL, " \t\n");
			}
			assert(nbsym < 10000);
			tbsym[nbsym++] = syms.add(p, mod, SYM_DEFINED, need, 0);
		}
		fclose(fp);
	} else { /* Use the exported symbols from currently running kernel */
		if (!get_kernel_info(K_SYMBOLS))
			return -1;

		for (ksym = ksyms; so_far < nksyms; ++so_far, ksym++) {
			assert(nbsym < 10000);
			tbsym[nbsym++] = syms.add((char *)ksym->name,
						   mod, SYM_DEFINED, need, 0);
		}
	}

	int size = nbsym * sizeof(SYMBOL *);
	mod->pub.tb = (SYMBOL **)xmalloc(size);
	memcpy(mod->pub.tb, tbsym, size);

	return 0;
}

/*
 *	Prints the dependancies in the output or stdout if depfile is NULL.
 */
static void depmod_prtdepend(MODULES & mods,
			     const char *pdepfile,
			     char *base_dir,
			     int verbose,
			     int quiet,
			     int showerror)
{
	FILE *out = stdout;

	if (pdepfile != NULL) {
		if ((out = fopen(pdepfile, "w")) == NULL) {
			error("Can't open %s", pdepfile);
			exit(-1);
		}
	}
	mods.prtdepend(out, "-", (base_dir ? strlen(base_dir):0), verbose, quiet, showerror);
}

static void usage()
{
	fprintf(stderr,
	"depmod " MODUTILS_VERSION "\n"
	"depmod -a [-n -e -v -q -V]\n"
	"      [-C configfile] [-F kernelsyms] [-b basedirectory] [forced_version]\n"
	"depmod [-n -e -v -q] [-f kernelsyms] module1.o module2.o ...\n"
	"If no arguments (expect options) are given, \"depmod -a\" is assumed\n"
	"\n"
	"depmod will output a dependancy list suitable for the modprobe utility.\n"
	"depmod -a will find the list of modules to probe from the file\n"
	ETC_CONF_MODULES ". It will output the result into the depfile specified\n"
	"in this configuration file\n"
	"\n"
	"Normally depmod operates silently, reporting only the list of modules that\n"
	"won't load properly (missing symbols).\n"
	"Option -q will make depmod keep quiet about missing symbols\n"
	"Option -e output all the unresolved symbol for a given module\n"
	"Option -s output all error message to the syslog daemon\n"
	"Option -v force a printout of all visited modules.\n"
	"Option -n will write the dependency file on stdout only.\n"
	"Option -V will show you the release version of depmod\n"
	"\n"
	"The following options are useful for people managing distributions;\n"
	"Option '-b basedirectory': use an image of a module tree.\n"
	"Option '-C configfile': use the file instead of /etc/conf.modules.\n"
	"Option '-F kernelsyms': use the file instead of the current kernel symbols.\n"
	);
}

int main(int argc, char *argv[])
{
	int ret = -1;
	int showerror = 0;
	int verbose = 0;
	int stdmode = 0;
	int err = 0;
	int nflag = 0;
	int o;
	int quiet = 0;
	char *conf_file = NULL;
	char *file_syms = NULL;
	char *base_dir = NULL;
	MODULES mods;
	SYMBOLS syms;

	struct option long_opts[] = {
		{"all", 0, 0, 'a'},
		{"basedir", 1, 0, 'b'},
		{"config", 1, 0, 'C'},
		{"errsyms", 0, 0, 'e'},
		{"filesysm", 1, 0, 'F'},
		{"help", 0, 0, 'h'},
		{"show", 0, 0, 'n'},
		{"quiet", 0, 0, 'q'},
		{"syslog", 0, 0, 's'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{0, 0, 0, 0 }
	};

	while ((o = getopt_long(argc, argv, "ab:C:eF:hnqsvV",
				&long_opts[0], NULL)) != EOF) {
		switch (o) {
		case 'a':
			stdmode = 1;	/* Probe standard directory */
			break;		/* using the config file */

		case 'e':
			showerror = 1;
			break;

		case 'b':
			base_dir = optarg;
			break;

		case '?':
		case 'h':
			usage();
			return 0;
			break;

		case 'C':
			conf_file = optarg;
			break;

		case 'F':
			file_syms = optarg;
			break;

		case 'n':
			nflag = 1;
			break;

		case 'q':
			quiet = 1;
			break;

		case 's':
			setsyslog("depmod");
			break;

		case 'v':
			verbose = 1;
			break;

		case 'V':
			printf("depmod version %s\n", MODUTILS_VERSION);
			break;

		default:
			err = 1;
			break;
		}
	}

	if (err) {
		error("Aborting");
		return -1;
	}
	/* else */
	argc -= optind;
	argv += optind;

	if (stdmode || argc == 0) {
		/* option -a is the default without arguments */
		if (argc > 0) {
			if (config_read(0, *argv, base_dir, conf_file) == -1) {
				error("%s does not exist", depfile);
				return -1;
			}
		} else if (config_read(0, NULL, base_dir, conf_file) == -1) {
			return -1;
		}

		int i;
		GLOB_LIST *g = config_lstmod(NULL, NULL, 0);
		for (i = 0, ret = 0; i < g->pathc && ret != -1; i++) {
			ret = mods.loadobj(syms, g->pathv[i]);
			if (ret)
				ret = 0;
		}
		if (ret != -1 &&
		    (ret = depmod_addksyms(mods, syms, file_syms)) != -1) {
			if (nflag)
				depmod_prtdepend(mods, NULL,
					base_dir, verbose, quiet, showerror);
			else
				depmod_prtdepend(mods, depfile,
					base_dir, verbose, quiet, showerror);
		}
	} else {
		/* not stdmode */
		for (ret = 0; argc > 0 && ret != -1; ++argv, --argc)
			mods.loadobj(syms, *argv);
		if (ret != -1 &&
		    (ret = depmod_addksyms(mods, syms, file_syms)) != -1) {
				depmod_prtdepend(mods, NULL,
					base_dir, verbose, quiet, showerror);
		}
	}

	return ret;
}
