/* SPDX-License-Identifier: GPL-2.0 */
/* Copyright(c) 2013 - 2018 Intel Corporation. */

/* Modeled on trace-events-sample.h */

/* The trace subsystem name for iavf will be "iavf".
 *
 * This file is named iavf_trace.h.
 *
 * Since this include file's name is different from the trace
 * subsystem name, we'll have to define TRACE_INCLUDE_FILE at the end
 * of this file.
 */
#undef TRACE_SYSTEM
#define TRACE_SYSTEM iavf

/* See trace-events-sample.h for a detailed description of why this
 * guard clause is different from most normal include files.
 */
#if !defined(_IAVF_TRACE_H_) || defined(TRACE_HEADER_MULTI_READ)
#define _IAVF_TRACE_H_

#include <linux/tracepoint.h>

/**
 * iavf_trace() macro enables shared code to refer to trace points
 * like:
 *
 * trace_iavf{,vf}_example(args...)
 *
 * ... as:
 *
 * iavf_trace(example, args...)
 *
 * ... to resolve to the PF or VF version of the tracepoint without
 * ifdefs, and to allow tracepoints to be disabled entirely at build
 * time.
 *
 * Trace point should always be referred to in the driver via this
 * macro.
 *
 * Similarly, iavf_trace_enabled(trace_name) wraps references to
 * trace_iavf{,vf}_<trace_name>_enabled() functions.
 */
#define _IAVF_TRACE_NAME(trace_name) (trace_ ## iavf ## _ ## trace_name)
#define IAVF_TRACE_NAME(trace_name) _IAVF_TRACE_NAME(trace_name)

#define iavf_trace(trace_name, args...) IAVF_TRACE_NAME(trace_name)(args)

#define iavf_trace_enabled(trace_name) IAVF_TRACE_NAME(trace_name##_enabled)()

/* Events common to PF and VF. Corresponding versions will be defined
 * for both, named trace_iavf_* and trace_iavf_*. The iavf_trace()
 * macro above will select the right trace point name for the driver
 * being built from shared code.
 */

/* Events related to a vsi & ring */
DECLARE_EVENT_CLASS(
	iavf_tx_template,

	TP_PROTO(struct iavf_ring *ring,
		 struct iavf_tx_desc *desc,
		 struct iavf_tx_buffer *buf),

	TP_ARGS(ring, desc, buf),

	/* The convention here is to make the first fields in the
	 * TP_STRUCT match the TP_PROTO exactly. This enables the use
	 * of the args struct generated by the tplist tool (from the
	 * bcc-tools package) to be used for those fields. To access
	 * fields other than the tracepoint args will require the
	 * tplist output to be adjusted.
	 */
	TP_STRUCT__entry(
		__field(void*, ring)
		__field(void*, desc)
		__field(void*, buf)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->ring = ring;
		__entry->desc = desc;
		__entry->buf = buf;
		__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s ring: %p desc: %p buf %p",
		__get_str(devname), __entry->ring,
		__entry->desc, __entry->buf)
);

DEFINE_EVENT(
	iavf_tx_template, iavf_clean_tx_irq,
	TP_PROTO(struct iavf_ring *ring,
		 struct iavf_tx_desc *desc,
		 struct iavf_tx_buffer *buf),

	TP_ARGS(ring, desc, buf));

DEFINE_EVENT(
	iavf_tx_template, iavf_clean_tx_irq_unmap,
	TP_PROTO(struct iavf_ring *ring,
		 struct iavf_tx_desc *desc,
		 struct iavf_tx_buffer *buf),

	TP_ARGS(ring, desc, buf));

DECLARE_EVENT_CLASS(
	iavf_rx_template,

	TP_PROTO(struct iavf_ring *ring,
		 union iavf_32byte_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb),

	TP_STRUCT__entry(
		__field(void*, ring)
		__field(void*, desc)
		__field(void*, skb)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->ring = ring;
		__entry->desc = desc;
		__entry->skb = skb;
		__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s ring: %p desc: %p skb %p",
		__get_str(devname), __entry->ring,
		__entry->desc, __entry->skb)
);

DEFINE_EVENT(
	iavf_rx_template, iavf_clean_rx_irq,
	TP_PROTO(struct iavf_ring *ring,
		 union iavf_32byte_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb));

DEFINE_EVENT(
	iavf_rx_template, iavf_clean_rx_irq_rx,
	TP_PROTO(struct iavf_ring *ring,
		 union iavf_32byte_rx_desc *desc,
		 struct sk_buff *skb),

	TP_ARGS(ring, desc, skb));

DECLARE_EVENT_CLASS(
	iavf_xmit_template,

	TP_PROTO(struct sk_buff *skb,
		 struct iavf_ring *ring),

	TP_ARGS(skb, ring),

	TP_STRUCT__entry(
		__field(void*, skb)
		__field(void*, ring)
		__string(devname, ring->netdev->name)
	),

	TP_fast_assign(
		__entry->skb = skb;
		__entry->ring = ring;
		__assign_str(devname, ring->netdev->name);
	),

	TP_printk(
		"netdev: %s skb: %p ring: %p",
		__get_str(devname), __entry->skb,
		__entry->ring)
);

DEFINE_EVENT(
	iavf_xmit_template, iavf_xmit_frame_ring,
	TP_PROTO(struct sk_buff *skb,
		 struct iavf_ring *ring),

	TP_ARGS(skb, ring));

DEFINE_EVENT(
	iavf_xmit_template, iavf_xmit_frame_ring_drop,
	TP_PROTO(struct sk_buff *skb,
		 struct iavf_ring *ring),

	TP_ARGS(skb, ring));

/* Events unique to the VF. */

#endif /* _IAVF_TRACE_H_ */
/* This must be outside ifdef _IAVF_TRACE_H */

/* This trace include file is not located in the .../include/trace
 * with the kernel tracepoint definitions, because we're a loadable
 * module.
 */
#undef TRACE_INCLUDE_PATH
#define TRACE_INCLUDE_PATH .
#undef TRACE_INCLUDE_FILE
#define TRACE_INCLUDE_FILE iavf_trace
#include <trace/define_trace.h>
