/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Cedrus VPU driver
 *
 * Copyright (c) 2013-2016 Jens Kuske <jenskuske@gmail.com>
 * Copyright (C) 2016 Florent Revest <florent.revest@free-electrons.com>
 * Copyright (C) 2018 Paul Kocialkowski <paul.kocialkowski@bootlin.com>
 */

#ifndef _CEDRUS_REGS_H_
#define _CEDRUS_REGS_H_

/*
 * Common acronyms and contractions used in register descriptions:
 * * VLD : Variable-Length Decoder
 * * IQ: Inverse Quantization
 * * IDCT: Inverse Discrete Cosine Transform
 * * MC: Motion Compensation
 * * STCD: Start Code Detect
 * * SDRT: Scale Down and Rotate
 */

#define VE_ENGINE_DEC_MPEG			0x100
#define VE_ENGINE_DEC_H264			0x200

#define VE_MODE					0x00

#define VE_MODE_REC_WR_MODE_2MB			(0x01 << 20)
#define VE_MODE_REC_WR_MODE_1MB			(0x00 << 20)
#define VE_MODE_DDR_MODE_BW_128			(0x03 << 16)
#define VE_MODE_DDR_MODE_BW_256			(0x02 << 16)
#define VE_MODE_DISABLED			(0x07 << 0)
#define VE_MODE_DEC_H265			(0x04 << 0)
#define VE_MODE_DEC_H264			(0x01 << 0)
#define VE_MODE_DEC_MPEG			(0x00 << 0)

#define VE_PRIMARY_CHROMA_BUF_LEN		0xc4
#define VE_PRIMARY_FB_LINE_STRIDE		0xc8

#define VE_PRIMARY_FB_LINE_STRIDE_CHROMA(s)	(((s) << 16) & GENMASK(31, 16))
#define VE_PRIMARY_FB_LINE_STRIDE_LUMA(s)	(((s) << 0) & GENMASK(15, 0))

#define VE_CHROMA_BUF_LEN			0xe8

#define VE_SECONDARY_OUT_FMT_TILED_32_NV12	(0x00 << 30)
#define VE_SECONDARY_OUT_FMT_EXT		(0x01 << 30)
#define VE_SECONDARY_OUT_FMT_YU12		(0x02 << 30)
#define VE_SECONDARY_OUT_FMT_YV12		(0x03 << 30)
#define VE_CHROMA_BUF_LEN_SDRT(l)		((l) & GENMASK(27, 0))

#define VE_PRIMARY_OUT_FMT			0xec

#define VE_PRIMARY_OUT_FMT_TILED_32_NV12	(0x00 << 4)
#define VE_PRIMARY_OUT_FMT_TILED_128_NV12	(0x01 << 4)
#define VE_PRIMARY_OUT_FMT_YU12			(0x02 << 4)
#define VE_PRIMARY_OUT_FMT_YV12			(0x03 << 4)
#define VE_PRIMARY_OUT_FMT_NV12			(0x04 << 4)
#define VE_PRIMARY_OUT_FMT_NV21			(0x05 << 4)
#define VE_SECONDARY_OUT_FMT_EXT_TILED_32_NV12	(0x00 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_TILED_128_NV12	(0x01 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_YU12		(0x02 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_YV12		(0x03 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_NV12		(0x04 << 0)
#define VE_SECONDARY_OUT_FMT_EXT_NV21		(0x05 << 0)

#define VE_VERSION				0xf0

#define VE_VERSION_SHIFT			16

#define VE_DEC_MPEG_MP12HDR			(VE_ENGINE_DEC_MPEG + 0x00)

#define VE_DEC_MPEG_MP12HDR_SLICE_TYPE(t)	(((t) << 28) & GENMASK(30, 28))
#define VE_DEC_MPEG_MP12HDR_F_CODE_SHIFT(x, y)	(24 - 4 * (y) - 8 * (x))
#define VE_DEC_MPEG_MP12HDR_F_CODE(__x, __y, __v) \
	(((__v) & GENMASK(3, 0)) << VE_DEC_MPEG_MP12HDR_F_CODE_SHIFT(__x, __y))

#define VE_DEC_MPEG_MP12HDR_INTRA_DC_PRECISION(p) \
	(((p) << 10) & GENMASK(11, 10))
#define VE_DEC_MPEG_MP12HDR_INTRA_PICTURE_STRUCTURE(s) \
	(((s) << 8) & GENMASK(9, 8))
#define VE_DEC_MPEG_MP12HDR_TOP_FIELD_FIRST(v) \
	((v) ? BIT(7) : 0)
#define VE_DEC_MPEG_MP12HDR_FRAME_PRED_FRAME_DCT(v) \
	((v) ? BIT(6) : 0)
#define VE_DEC_MPEG_MP12HDR_CONCEALMENT_MOTION_VECTORS(v) \
	((v) ? BIT(5) : 0)
#define VE_DEC_MPEG_MP12HDR_Q_SCALE_TYPE(v) \
	((v) ? BIT(4) : 0)
#define VE_DEC_MPEG_MP12HDR_INTRA_VLC_FORMAT(v) \
	((v) ? BIT(3) : 0)
#define VE_DEC_MPEG_MP12HDR_ALTERNATE_SCAN(v) \
	((v) ? BIT(2) : 0)
#define VE_DEC_MPEG_MP12HDR_FULL_PEL_FORWARD_VECTOR(v) \
	((v) ? BIT(1) : 0)
#define VE_DEC_MPEG_MP12HDR_FULL_PEL_BACKWARD_VECTOR(v) \
	((v) ? BIT(0) : 0)

#define VE_DEC_MPEG_PICCODEDSIZE		(VE_ENGINE_DEC_MPEG + 0x08)

#define VE_DEC_MPEG_PICCODEDSIZE_WIDTH(w) \
	((DIV_ROUND_UP((w), 16) << 8) & GENMASK(15, 8))
#define VE_DEC_MPEG_PICCODEDSIZE_HEIGHT(h) \
	((DIV_ROUND_UP((h), 16) << 0) & GENMASK(7, 0))

#define VE_DEC_MPEG_PICBOUNDSIZE		(VE_ENGINE_DEC_MPEG + 0x0c)

#define VE_DEC_MPEG_PICBOUNDSIZE_WIDTH(w)	(((w) << 16) & GENMASK(27, 16))
#define VE_DEC_MPEG_PICBOUNDSIZE_HEIGHT(h)	(((h) << 0) & GENMASK(11, 0))

#define VE_DEC_MPEG_MBADDR			(VE_ENGINE_DEC_MPEG + 0x10)

#define VE_DEC_MPEG_MBADDR_X(w)			(((w) << 8) & GENMASK(15, 8))
#define VE_DEC_MPEG_MBADDR_Y(h)			(((h) << 0) & GENMASK(0, 7))

#define VE_DEC_MPEG_CTRL			(VE_ENGINE_DEC_MPEG + 0x14)

#define VE_DEC_MPEG_CTRL_MC_CACHE_EN		BIT(31)
#define VE_DEC_MPEG_CTRL_SW_VLD			BIT(27)
#define VE_DEC_MPEG_CTRL_SW_IQ_IS		BIT(17)
#define VE_DEC_MPEG_CTRL_QP_AC_DC_OUT_EN	BIT(14)
#define VE_DEC_MPEG_CTRL_ROTATE_SCALE_OUT_EN	BIT(8)
#define VE_DEC_MPEG_CTRL_MC_NO_WRITEBACK	BIT(7)
#define VE_DEC_MPEG_CTRL_ROTATE_IRQ_EN		BIT(6)
#define VE_DEC_MPEG_CTRL_VLD_DATA_REQ_IRQ_EN	BIT(5)
#define VE_DEC_MPEG_CTRL_ERROR_IRQ_EN		BIT(4)
#define VE_DEC_MPEG_CTRL_FINISH_IRQ_EN		BIT(3)
#define VE_DEC_MPEG_CTRL_IRQ_MASK \
	(VE_DEC_MPEG_CTRL_FINISH_IRQ_EN | VE_DEC_MPEG_CTRL_ERROR_IRQ_EN | \
	 VE_DEC_MPEG_CTRL_VLD_DATA_REQ_IRQ_EN)

#define VE_DEC_MPEG_TRIGGER			(VE_ENGINE_DEC_MPEG + 0x18)

#define VE_DEC_MPEG_TRIGGER_MB_BOUNDARY		BIT(31)

#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_420	(0x00 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_411	(0x01 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_422	(0x02 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_444	(0x03 << 27)
#define VE_DEC_MPEG_TRIGGER_CHROMA_FMT_422T	(0x04 << 27)

#define VE_DEC_MPEG_TRIGGER_MPEG1		(0x01 << 24)
#define VE_DEC_MPEG_TRIGGER_MPEG2		(0x02 << 24)
#define VE_DEC_MPEG_TRIGGER_JPEG		(0x03 << 24)
#define VE_DEC_MPEG_TRIGGER_MPEG4		(0x04 << 24)
#define VE_DEC_MPEG_TRIGGER_VP62		(0x05 << 24)

#define VE_DEC_MPEG_TRIGGER_VP62_AC_GET_BITS	BIT(7)

#define VE_DEC_MPEG_TRIGGER_STCD_VC1		(0x02 << 4)
#define VE_DEC_MPEG_TRIGGER_STCD_MPEG2		(0x01 << 4)
#define VE_DEC_MPEG_TRIGGER_STCD_AVC		(0x00 << 4)

#define VE_DEC_MPEG_TRIGGER_HW_MPEG_VLD		(0x0f << 0)
#define VE_DEC_MPEG_TRIGGER_HW_JPEG_VLD		(0x0e << 0)
#define VE_DEC_MPEG_TRIGGER_HW_MB		(0x0d << 0)
#define VE_DEC_MPEG_TRIGGER_HW_ROTATE		(0x0c << 0)
#define VE_DEC_MPEG_TRIGGER_HW_VP6_VLD		(0x0b << 0)
#define VE_DEC_MPEG_TRIGGER_HW_MAF		(0x0a << 0)
#define VE_DEC_MPEG_TRIGGER_HW_STCD_END		(0x09 << 0)
#define VE_DEC_MPEG_TRIGGER_HW_STCD_BEGIN	(0x08 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_MC		(0x07 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_IQ		(0x06 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_IDCT		(0x05 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_SCALE		(0x04 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_VP6		(0x03 << 0)
#define VE_DEC_MPEG_TRIGGER_SW_VP62_AC_GET_BITS	(0x02 << 0)

#define VE_DEC_MPEG_STATUS			(VE_ENGINE_DEC_MPEG + 0x1c)

#define VE_DEC_MPEG_STATUS_START_DETECT_BUSY	BIT(27)
#define VE_DEC_MPEG_STATUS_VP6_BIT		BIT(26)
#define VE_DEC_MPEG_STATUS_VP6_BIT_BUSY		BIT(25)
#define VE_DEC_MPEG_STATUS_MAF_BUSY		BIT(23)
#define VE_DEC_MPEG_STATUS_VP6_MVP_BUSY		BIT(22)
#define VE_DEC_MPEG_STATUS_JPEG_BIT_END		BIT(21)
#define VE_DEC_MPEG_STATUS_JPEG_RESTART_ERROR	BIT(20)
#define VE_DEC_MPEG_STATUS_JPEG_MARKER		BIT(19)
#define VE_DEC_MPEG_STATUS_ROTATE_BUSY		BIT(18)
#define VE_DEC_MPEG_STATUS_DEBLOCKING_BUSY	BIT(17)
#define VE_DEC_MPEG_STATUS_SCALE_DOWN_BUSY	BIT(16)
#define VE_DEC_MPEG_STATUS_IQIS_BUF_EMPTY	BIT(15)
#define VE_DEC_MPEG_STATUS_IDCT_BUF_EMPTY	BIT(14)
#define VE_DEC_MPEG_STATUS_VE_BUSY		BIT(13)
#define VE_DEC_MPEG_STATUS_MC_BUSY		BIT(12)
#define VE_DEC_MPEG_STATUS_IDCT_BUSY		BIT(11)
#define VE_DEC_MPEG_STATUS_IQIS_BUSY		BIT(10)
#define VE_DEC_MPEG_STATUS_DCAC_BUSY		BIT(9)
#define VE_DEC_MPEG_STATUS_VLD_BUSY		BIT(8)
#define VE_DEC_MPEG_STATUS_ROTATE_SUCCESS	BIT(3)
#define VE_DEC_MPEG_STATUS_VLD_DATA_REQ		BIT(2)
#define VE_DEC_MPEG_STATUS_ERROR		BIT(1)
#define VE_DEC_MPEG_STATUS_SUCCESS		BIT(0)
#define VE_DEC_MPEG_STATUS_CHECK_MASK \
	(VE_DEC_MPEG_STATUS_SUCCESS | VE_DEC_MPEG_STATUS_ERROR | \
	 VE_DEC_MPEG_STATUS_VLD_DATA_REQ)
#define VE_DEC_MPEG_STATUS_CHECK_ERROR \
	(VE_DEC_MPEG_STATUS_ERROR | VE_DEC_MPEG_STATUS_VLD_DATA_REQ)

#define VE_DEC_MPEG_VLD_ADDR			(VE_ENGINE_DEC_MPEG + 0x28)

#define VE_DEC_MPEG_VLD_ADDR_FIRST_PIC_DATA	BIT(30)
#define VE_DEC_MPEG_VLD_ADDR_LAST_PIC_DATA	BIT(29)
#define VE_DEC_MPEG_VLD_ADDR_VALID_PIC_DATA	BIT(28)
#define VE_DEC_MPEG_VLD_ADDR_BASE(a)					\
	({								\
		u32 _tmp = (a);						\
		u32 _lo = _tmp & GENMASK(27, 4);			\
		u32 _hi = (_tmp >> 28) & GENMASK(3, 0);			\
		(_lo | _hi);						\
	})

#define VE_DEC_MPEG_VLD_OFFSET			(VE_ENGINE_DEC_MPEG + 0x2c)
#define VE_DEC_MPEG_VLD_LEN			(VE_ENGINE_DEC_MPEG + 0x30)
#define VE_DEC_MPEG_VLD_END_ADDR		(VE_ENGINE_DEC_MPEG + 0x34)

#define VE_DEC_MPEG_REC_LUMA			(VE_ENGINE_DEC_MPEG + 0x48)
#define VE_DEC_MPEG_REC_CHROMA			(VE_ENGINE_DEC_MPEG + 0x4c)
#define VE_DEC_MPEG_FWD_REF_LUMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x50)
#define VE_DEC_MPEG_FWD_REF_CHROMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x54)
#define VE_DEC_MPEG_BWD_REF_LUMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x58)
#define VE_DEC_MPEG_BWD_REF_CHROMA_ADDR		(VE_ENGINE_DEC_MPEG + 0x5c)

#define VE_DEC_MPEG_IQMINPUT			(VE_ENGINE_DEC_MPEG + 0x80)

#define VE_DEC_MPEG_IQMINPUT_FLAG_INTRA		(0x01 << 14)
#define VE_DEC_MPEG_IQMINPUT_FLAG_NON_INTRA	(0x00 << 14)
#define VE_DEC_MPEG_IQMINPUT_WEIGHT(i, v) \
	(((v) & GENMASK(7, 0)) | (((i) << 8) & GENMASK(13, 8)))

#define VE_DEC_MPEG_ERROR			(VE_ENGINE_DEC_MPEG + 0xc4)
#define VE_DEC_MPEG_CRTMBADDR			(VE_ENGINE_DEC_MPEG + 0xc8)
#define VE_DEC_MPEG_ROT_LUMA			(VE_ENGINE_DEC_MPEG + 0xcc)
#define VE_DEC_MPEG_ROT_CHROMA			(VE_ENGINE_DEC_MPEG + 0xd0)

#endif
