// SPDX-License-Identifier: GPL-2.0-only
/**
 * Routines supporting VMX instructions on the Power 8
 *
 * Copyright (C) 2015 International Business Machines Inc.
 *
 * Author: Marcelo Henrique Cerri <mhcerri@br.ibm.com>
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/err.h>
#include <linux/cpufeature.h>
#include <linux/crypto.h>
#include <asm/cputable.h>
#include <crypto/internal/hash.h>

extern struct shash_alg p8_ghash_alg;
extern struct crypto_alg p8_aes_alg;
extern struct crypto_alg p8_aes_cbc_alg;
extern struct crypto_alg p8_aes_ctr_alg;
extern struct crypto_alg p8_aes_xts_alg;
static struct crypto_alg *algs[] = {
	&p8_aes_alg,
	&p8_aes_cbc_alg,
	&p8_aes_ctr_alg,
	&p8_aes_xts_alg,
	NULL,
};

static int __init p8_init(void)
{
	int ret = 0;
	struct crypto_alg **alg_it;

	for (alg_it = algs; *alg_it; alg_it++) {
		ret = crypto_register_alg(*alg_it);
		printk(KERN_INFO "crypto_register_alg '%s' = %d\n",
		       (*alg_it)->cra_name, ret);
		if (ret) {
			for (alg_it--; alg_it >= algs; alg_it--)
				crypto_unregister_alg(*alg_it);
			break;
		}
	}
	if (ret)
		return ret;

	ret = crypto_register_shash(&p8_ghash_alg);
	if (ret) {
		for (alg_it = algs; *alg_it; alg_it++)
			crypto_unregister_alg(*alg_it);
	}
	return ret;
}

static void __exit p8_exit(void)
{
	struct crypto_alg **alg_it;

	for (alg_it = algs; *alg_it; alg_it++) {
		printk(KERN_INFO "Removing '%s'\n", (*alg_it)->cra_name);
		crypto_unregister_alg(*alg_it);
	}
	crypto_unregister_shash(&p8_ghash_alg);
}

module_cpu_feature_match(PPC_MODULE_FEATURE_VEC_CRYPTO, p8_init);
module_exit(p8_exit);

MODULE_AUTHOR("Marcelo Cerri<mhcerri@br.ibm.com>");
MODULE_DESCRIPTION("IBM VMX cryptographic acceleration instructions "
		   "support on Power 8");
MODULE_LICENSE("GPL");
MODULE_VERSION("1.0.0");
