#include <stdio.h>
#include <stdlib.h>
#include <elf.h>
#include ELF_MACHINE_H

#include "link.h"

#ifndef ElfW
# if ELFCLASSM == ELFCLASS32
#  define ElfW(x)  Elf32_ ## x
#  define ELFW(x)  ELF32_ ## x
# else
#  define ElfW(x)  Elf64_ ## x
#  define ELFW(x)  ELF64_ ## x
# endif
#endif


/*======================================================================*/

static void 
setupmod(MODULE *mod, SYMBOL *tbpub[], int nbpub, SYMBOL *tbext[], int nbext)
{
  mod->is_load = 1;
  mod->pub.nb = nbpub;
  mod->ext.nb = nbext;
  if (nbpub > 0)
    {
      int size = nbpub*sizeof(SYMBOL*);
      mod->pub.tb = (SYMBOL**)malloc_err(size,1);
      memcpy (mod->pub.tb,tbpub,size);
    }
  else
    {
      mod->pub.tb = NULL;
    }
  if (nbext > 0)
    {
      int size = nbext*sizeof(SYMBOL*);
      mod->ext.tb = (SYMBOL**)malloc_err(size,1);
      memcpy (mod->ext.tb,tbext,size);
      SYMBOL **ptext = mod->ext.tb;
      for (int i=0; i<nbext; i++,ptext++)
	(*ptext)->requis = 1;
    }
  else
    {
      mod->ext.tb = NULL;
    }
}

static char *
load_elf(FILE *fp, ElfW(Ehdr) *epnt, SYMBOLS &syms, MODULE *mod)
{
  SYMBOL *tbext[5000];
  int nbext = 0;
  SYMBOL *tbpub[5000];
  int nbpub = 0;
  int mr = 0;
  ElfW(Shdr) *sections;
  ElfW(Shdr) *spnt;
  ElfW(Sym) *sp;
  ElfW(Sym) *symtab = NULL;
  long filesize;
  char *stringtab = (char *)0;
  int i;
  int n;
  int nsymbols = 0;

  if (epnt->e_phnum != 0)
    return "can't handle program headers...";

  fseek(fp, 0L, SEEK_END);
  filesize = ftell(fp);

  /* load the section headers */
  sections = (ElfW(Shdr)*)malloc_err(epnt->e_shentsize * epnt->e_shnum, 1);

  fseek(fp, epnt->e_shoff, SEEK_SET);
  fread((char *)sections, epnt->e_shentsize * epnt->e_shnum, 1, fp);
  if (feof(fp) || ferror(fp))
    {
      free(sections);
      return "Error reading ELF section headers";
    }

  for (spnt = sections, i = 0; i < epnt->e_shnum; ++i, ++spnt)
    {
      switch (spnt->sh_type)
	{
	default:
	  break; /* IGNORE */

	case SHT_SYMTAB:
	  if (nsymbols)
	    {
	      free(sections);
	      if (stringtab)
		free(stringtab);
	      return "Can't handle >1 symbol section";
	    }
	  nsymbols = spnt->sh_size / spnt->sh_entsize;
	  symtab = (ElfW(Sym)*)malloc_err(nsymbols * sizeof (*symtab), 1);
	  fseek(fp, spnt->sh_offset, SEEK_SET);
	  fread(symtab, nsymbols * sizeof (*symtab), 1, fp);
	  if (feof(fp) || ferror(fp))
	    {
	      free(sections);
	      free(symtab);
	      if (stringtab)
		free(stringtab);
	      return "Error reading ELF SYMTAB section";
	    }
	  break;

	case SHT_STRTAB:
	  stringtab = (char *)malloc_err(spnt->sh_size, 1);
	  fseek(fp, spnt->sh_offset, SEEK_SET);
	  fread(stringtab, spnt->sh_size, 1, fp);
	  if (feof(fp) || ferror(fp))
	    {
	      free(sections);
	      free(stringtab);
	      if (symtab)
		free(symtab);
	      return "Error reading ELF STRTAB section";
	    }
	  break;
	}
    }

  for (n = nsymbols, sp = symtab ; --n >= 0 ; sp++)
    {
      char *name = stringtab + sp->st_name;

      if (*name == '\0')
	continue;
      if (sp->st_shndx == SHN_UNDEF)
	tbext[nbext++] = syms.add(name,NULL, SYM_REQUIS, mr, 0);
      else
	tbpub[nbpub++] = syms.add(name,mod, SYM_DEFINI,
				  mr, (sp->st_shndx == SHN_COMMON));
    }

  setupmod(mod, tbpub, nbpub, tbext, nbext);
  if (stringtab)
    free(stringtab);
  if (symtab)
    free(symtab);
  free(sections);
  return (char *)0;
}

int
load_obj(SYMBOLS &syms, const char *module, MODULE *mod)
{
  FILE *fp;
  ElfW(Ehdr) header;
  char *errstr;

  /* open file and read header */
  if ((fp = fopen(module, "r")) == NULL)
    return 1;

  fread(&header, sizeof(ElfW(Ehdr)), 1, fp);
  if (feof(fp) || ferror(fp))
    return 1;

  errstr = load_elf(fp, &header, syms, mod);

  if (errstr != (char *)0)
    return 1;
  return 0;
}
