/*
 * Universal Host Controller Interface driver for USB.
 *
 * (C) Copyright 1999 Linus Torvalds
 *
 * Intel documents this fairly well, and as far as I know there
 * are no royalties or anything like that, but even so there are
 * people who decided that they want to do the same thing in a
 * completely different way.
 *
 * Oh, well. The intel version is the more common by far. As such,
 * that's the one I care about right now.
 *
 * WARNING! The USB documentation is downright evil. Most of it
 * is just crap, written by a committee. You're better off ignoring
 * most of it, the important stuff is:
 *  - the low-level protocol (fairly simple but lots of small details)
 *  - working around the horridness of the rest
 */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>

#include <asm/spinlock.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>

#include "uhci.h"

#define compile_assert(x) do { switch (0) { case 1: case !(x): } } while (0)

static struct wait_queue *uhci_configure = NULL;

static void wait_ms(unsigned int ms)
{
	current->state = TASK_UNINTERRUPTIBLE;
	schedule_timeout(1 + ms / 10);
}

/*
 * Return the result of a TD..
 */
static int uhci_td_result(struct uhci_device * dev, struct uhci_td *td)
{
	unsigned int status;

	status = (td->status >> 16) & 0xff;

	/* Some debugging code */
	if (status) {
		int i = 10;
		struct uhci_td *tmp = dev->control_td;
		printk("uhci_td_result() failed with status %d\n", status);
		show_status(dev->uhci);
		do {
			show_td(tmp);
			tmp++;
			if (!--i)
				break;
		} while (tmp <= td);
	}
	return status;		
}

/*
 * UHCI interrupt list operations..
 */
static spinlock_t irqlist_lock = SPIN_LOCK_UNLOCKED;

static void uhci_add_irq_list(struct uhci *uhci, struct uhci_td *td, int (*completed)(int, void *))
{
	unsigned long flags;

	td->completed = completed;

	spin_lock_irqsave(&irqlist_lock, flags);
	list_add(&td->irq_list, &uhci->interrupt_list);
	spin_unlock_irqrestore(&irqlist_lock, flags);
}

static void uhci_remove_irq_list(struct uhci_td *td)
{
	unsigned long flags;

	spin_lock_irqsave(&irqlist_lock, flags);
	list_del(&td->irq_list);
	spin_unlock_irqrestore(&irqlist_lock, flags);
}

/*
 * Request a interrupt handler..
 */
static int uhci_request_irq(struct usb_device * usb, unsigned int pipe, int (*handler)(int, void *), int period)
{
	struct uhci_device *dev = usb_to_uhci(usb);
	struct uhci_td *td = dev->td + 0;
	struct uhci_qh *interrupt_qh = &dev->uhci->root_hub->interrupt8_qh;	/* Ignore the period for now */

	unsigned int destination, status;

	/* Destination: pipe destination with INPUT */
	destination = (pipe & 0x0007ff00)  |  0x69;

	/* Status:    slow/fast,      Interrupt,   Active,    Short Packet Detect     Infinite Errors */
	status = (pipe & (1 << 26)) | (1 << 24) | (1 << 23)   |   (1 << 29)       |    (0 << 27);

	td->link = 1;
	td->status = status;			/* In */
	td->info = destination | (7 << 21);	/* 8 bytes of data */
	td->buffer = virt_to_bus(dev->data);

	uhci_add_irq_list(dev->uhci, td, handler);
	wmb();
	interrupt_qh->element = virt_to_bus(td);
	return 0;
}

/*
 * Control thread operations: we just mark the last TD
 * in a control thread as an interrupt TD, and wake up
 * the front-end on completion.
 *
 * We need to remove the TD from the lists (both interrupt
 * list and TD lists) by hand if something bad happens!
 */
static struct wait_queue *control_wakeup;

static int uhci_control_completed(int status, void *data)
{
	wake_up(&control_wakeup);
	return 0;			/* Don't re-instate */
}

static int uhci_run_control(struct uhci_device *dev, struct uhci_td *td)
{
	struct wait_queue wait = { current, NULL };

	current->state = TASK_UNINTERRUPTIBLE;
	add_wait_queue(&control_wakeup, &wait);

	uhci_add_irq_list(dev->uhci, td, uhci_control_completed);
	dev->uhci->root_hub->control_qh.element = virt_to_bus(dev->control_td);

	schedule_timeout(HZ/10);

	remove_wait_queue(&control_wakeup, &wait);

	/* Clean up in case it failed.. */
	uhci_remove_irq_list(td);
	dev->uhci->root_hub->control_qh.element = 1;

	return uhci_td_result(dev, td);
}

/*
 * Send or receive a control message on a pipe.
 *
 * Note that the "pipe" structure is set up to map
 * easily to the uhci destination fields.
 *
 * A control message is built up from three parts:
 *  - The command itself
 *  - [ optional ] data phase
 *  - Status complete phase
 *
 * The data phase can be an arbitrary number of TD's
 * although we currently had better not have more than
 * 29 TD's here (we have 31 TD's allocated for control
 * operations, and two of them are used for command and
 * status).
 *
 * 29 TD's is a minimum of 232 bytes worth of control
 * information, that's just ridiculously high. Most
 * control messages have just a few bytes of data.
 */
static int uhci_control_msg(struct usb_device *usb, unsigned int pipe, void *cmd, void *data, int len)
{
	struct uhci_device *dev = usb_to_uhci(usb);
	struct uhci_td *td = dev->control_td;
	unsigned long destination, status;

	/* The "pipe" thing contains the destination in bits 8--18, 0x2D is SETUP */
	destination = (pipe & 0x0007ff00) | 0x2D;

	/* Status:    slow/fast,       Active,    Short Packet Detect     Three Errors */
	status = (pipe & (1 << 26)) | (1 << 23)   |   (1 << 29)       |    (3 << 27);

	/*
	 * Build the TD for the control request
	 */
	td->link = 4 | virt_to_bus(td+1);		/* Point to next TD for input data */
	td->status = status;				/* Try forever */
	td->info = destination | (7 << 21);		/* 8 bytes of data */
	td->buffer = virt_to_bus(cmd);

	/*
	 * If direction is "send", change the frame from SETUP (0x2D)
	 * to OUT (0xE1). Else change it from SETUP to IN (0x69)
	 */
	destination ^= (0x2D ^ 0x69);			/* SETUP -> IN */
	if (usb_pipeout(pipe))
		destination ^= (0xE1 ^ 0x69);		/* IN -> OUT */

	td++;

	/*
	 * Build the DATA TD's
	 */
	while (len > 0) {
		/* Build the TD for control status */
		int pktsze = len;
		int maxsze = usb_maxpacket(pipe);

		if (pktsze > maxsze)
			pktsze = maxsze;

		/* Alternate Data0/1 (start with Data1) */
		destination ^= 1 << 19;
	
		td->link = 4 | virt_to_bus(td+1);			/* Point to next TD for data */
		td->status = status;					/* Status */
		td->info = destination | ((pktsze-1) << 21);		/* pktsze bytes of data */
		td->buffer = virt_to_bus(data);

		td++;
		data += maxsze;
		len -= maxsze;
	}

	/*
	 * Build the final TD for control status
	 */
	destination ^= (0xE1 ^ 0x69);			/* OUT -> IN */
	destination |= 1 << 19;				/* End in Data1 */

	td->link = 1;					/* Terminate */
	td->status = status | (1 << 24);		/* IOC */
	td->info = destination | (0x7ff << 21);		/* 0 bytes of data */
	td->buffer = 0;

	/* Start it up.. */
	return uhci_run_control(dev, td);
}

static int uhci_deallocate(struct usb_device *usb)
{
	struct uhci_device *dev = usb_to_uhci(usb);

	free_page((unsigned long) dev);
	return 0;
}

struct usb_operations uhci_device_operations = {
	uhci_control_msg,
	uhci_request_irq,
	uhci_deallocate,
};

static int uhci_root_control_msg(struct usb_device *usb, unsigned int pipe, void *cmd, void *data, int len)
{
	printk("What? Who tries to do a control operation on the root hub?\n");
	return -1;
}

static int uhci_root_deallocate(struct usb_device *usb)
{
	printk("What? Who tries to deallocate the root hub?\n");
	return -1;
}

static int uhci_root_request_irq(struct usb_device * usb, unsigned int pipe, int (*handler)(int, void *), int period)
{
	printk("What? Who tries to get interrupts from the root hub?\n");
	return -1;
}


struct usb_operations uhci_root_operations = {
	uhci_root_control_msg,
	uhci_root_request_irq,
	uhci_root_deallocate,
};

/*
 * This is just incredibly fragile. The timings must be just
 * right, and they aren't really documented very well.
 *
 * Note the short delay between disabling reset and enabling
 * the port..
 */
static void uhci_reset_port(unsigned int port)
{
	/*
	 * Ok, we're on top of it now..
	 *
	 * Enable reset for 10ms, then let it sit
	 * for 100ms. After this the device will
	 * be in the "default" state.
	 */
	outw(USBPORTSC_PR, port);
	wait_ms(50);
	outw(0, port);
	udelay(5);

	/* Enable the port.. */
	outw(USBPORTSC_PE | USBPORTSC_PEC | USBPORTSC_CSC, port);
	wait_ms(10);
}


/*
 * This gets called if the connect status on the root
 * hub (and the root hub only) changes.
 */
static void uhci_connect_change(struct uhci * uhci, unsigned int port, unsigned int nr)
{
	struct uhci_device * dev;
	unsigned short status;

	/*
	 * Even if the status says we're connected,
	 * the fact that the status bits changed may
	 * that we got disconnected and then reconnected.
	 *
	 * So start off by getting rid of any old devices..
	 */
	usb_disconnect(uhci->root_hub->usb.children + nr);

	status = inw(port);

	/* If we have nothing connected, then clear change status and disable the port */
	status = (status & ~USBPORTSC_PE) | USBPORTSC_PEC;
	if (!(status & USBPORTSC_CCS)) {
		outw(status, port);
		return;
	}

	/*
	 * Ok, we got a new connection. Allocate a device to it,
	 * and find out what it wants to do..
	 */
	compile_assert(sizeof(struct uhci_device) < PAGE_SIZE);
	dev = (struct uhci_device *) __get_free_page(GFP_KERNEL);
	if (!dev)
		return;

	/* Initialize "dev" */
	memset(dev, 0, sizeof(*dev));
	dev->uhci = uhci;
	usb_connect(&dev->usb, &uhci->devmap);
	dev->usb.op = &uhci_device_operations;

	uhci->root_hub->usb.children[nr] = &dev->usb;

	uhci_reset_port(port);

	/* Get speed information */
	dev->usb.slow = (inw(port) & USBPORTSC_LSDA) ? 1 : 0;

	/*
	 * Ok, all the stuff specific to the root hub has been done.
	 * The rest is generic for any new USB attach, regardless of
	 * hub type.
	 */
	usb_new_device(&dev->usb);
}

/*
 * This gets called when the root hub configuration
 * has changed. Just go through each port, seeing if
 * there is something interesting happening.
 */
static void uhci_check_configuration(struct uhci *uhci)
{
	unsigned int io_addr = uhci->io_addr + USBPORTSC1;
	int maxchild = uhci->root_hub->usb.maxchild;
	int nr = 0;

	do {
		unsigned short status = inw(io_addr);

		if (status & USBPORTSC_CSC)
			uhci_connect_change(uhci, io_addr, nr);

		nr++; io_addr += 2;
	} while (nr < maxchild);
}

static void uhci_interrupt_notify(struct uhci *uhci)
{
	struct list_head *head = &uhci->interrupt_list;
	struct list_head *tmp;

	spin_lock(&irqlist_lock);
	tmp = head->next;
	while (tmp != head) {
		struct uhci_td *td = list_entry(tmp, struct uhci_td, irq_list);
		struct list_head *next;

		next = tmp->next;

		if (!(td->status & (1 << 23))) {	/* No longer active? */
			__list_del(tmp->prev, next);
			INIT_LIST_HEAD(tmp);
			if (td->completed(td->status, bus_to_virt(td->buffer))) {
				struct uhci_qh * interrupt_qh = &uhci->root_hub->interrupt8_qh;
				list_add(&td->irq_list, &uhci->interrupt_list);
				td->status = (td->status & 0x2f000000) | (1 << 23) | (1 << 24);	/* active */
				wmb();
				interrupt_qh->element = virt_to_bus(td);
			}
		}
		tmp = next;
	}
	spin_unlock(&irqlist_lock);
}

/*
 * Check port status - Connect Status Change - for
 * each of the attached ports (defaults to two ports,
 * but at least in theory there can be more of them).
 *
 * Wake up the configurator if something happened, we
 * can't really do much at interrupt time.
 */
static void uhci_root_hub_events(struct uhci *uhci, unsigned int io_addr)
{
	if (waitqueue_active(&uhci_configure)) {
		int ports = uhci->root_hub->usb.maxchild;
		io_addr += USBPORTSC1;
		do {
			if (inw(io_addr) & USBPORTSC_CSC) {
				wake_up(&uhci_configure);
				return;
			}
			io_addr += 2;
		} while (--ports > 0);
	}
}

static void uhci_interrupt(int irq, void *__uhci, struct pt_regs *regs)
{
	struct uhci *uhci = __uhci;
	unsigned int io_addr = uhci->io_addr;
	unsigned short status;

	/*
	 * Read the interrupt status, and write it back to clear the interrupt cause
	 */
	status = inw(io_addr + USBSTS);
	outw(status, io_addr + USBSTS);

	/* Walk the list of pending TD's to see which ones completed.. */
	uhci_interrupt_notify(uhci);

	/* Check if there are any events on the root hub.. */
	uhci_root_hub_events(uhci, io_addr);
}

/*
 * We init one packet, and mark it just IOC and _not_
 * active. Which will result in no actual USB traffic,
 * but _will_ result in an interrupt every second.
 *
 * Which is exactly what we want.
 */
static void uhci_init_ticktd(struct uhci *uhci)
{
	struct uhci_device *dev = uhci->root_hub;
	struct uhci_td *td = &dev->tick_td;

	td->link = 1;
	td->status = (1 << 24);					/* interrupt on completion */
	td->info = (15 << 21) | 0x7f69;				/* (ignored) input packet, 16 bytes, device 127 */
	td->buffer = 0;
	uhci->fl->frame[0] = virt_to_bus(td);
}

static void reset_hc(struct uhci *uhci)
{
	unsigned int io_addr = uhci->io_addr;

	/* Global reset for 50ms */
	outw(USBCMD_GRESET, io_addr+USBCMD);
	wait_ms(50);
	outw(0, io_addr+USBCMD);
	wait_ms(10);
}

static void start_hc(struct uhci *uhci)
{
	unsigned int io_addr = uhci->io_addr;
	int timeout = 1000;

	uhci_init_ticktd(uhci);

	/*
	 * Reset the HC - this will force us to get a
	 * new notification of any already connected
	 * ports due to the virtual disconnect that it
	 * implies.
	 */
	outw(USBCMD_HCRESET, io_addr + USBCMD);
	while (inw(io_addr + USBCMD) & USBCMD_HCRESET) {
		if (!--timeout) {
			printk("USBCMD_HCRESET timed out!\n");
			break;
		}
	}

	outw(USBINTR_TIMEOUT | USBINTR_RESUME | USBINTR_IOC | USBINTR_SP, io_addr + USBINTR);
	outw(0, io_addr + USBFRNUM);
	outl(virt_to_bus(uhci->fl), io_addr + USBFLBASEADD);

	/* Run and mark it configured with a 64-byte max packet */
	outw(USBCMD_RS | USBCMD_CF, io_addr + USBCMD);
}

/*
 * Right now this driver only drives a single uhci,
 * so we allocate the structure statically. We should
 * eventually fix this (you can just load multiple
 * copies of the same module for multiple controllers,
 * though).
 *
 * We could allocate all of this dynamically, but
 * it's basically fairly rare to have multiple USB
 * buses.
 */
struct uhci static_uhci = {
	0,			/* IRQ - filled in by PCI probe */
	0,			/* IO Address - filled in by PCI probe */
	/* Device number bitmap */
	{ { 0, } },
};

/*
 * Allocate a frame list, and four regular queues.
 *
 * The hardware doesn't really know any difference
 * in the queues, but the order does matter for the
 * protocols higher up. The order is:
 *
 *  - any isochronous events handled before any
 *    of the queues. We don't do that here, because
 *    we'll create the actual TD entries on demand.
 *  - The first queue is the "interrupt queue".
 *  - The second queue is the "control queue".
 *  - The third queue is "bulk data".
 *
 * We could certainly have multiple queues of the same
 * type, and maybe we should. We could have per-device
 * queues, for example. We begin small.
 */
static struct uhci * alloc_uhci(unsigned int io_addr)
{
	int i;
	struct uhci *uhci = &static_uhci;
	struct uhci_device *hub;

	uhci->irq = -1;
	uhci->io_addr = io_addr;
	INIT_LIST_HEAD(&uhci->interrupt_list);

	/*
	 * We allocate a 8kB area for the UHCI hub. The area
	 * is described by the uhci_device structure, and basically
	 * contains everything needed for normal operation.
	 *
	 * The first page is the actual device descriptor for the
	 * hub.
	 *
	 * The second page is used for the frame list.
	 */
	hub = (struct uhci_device *) __get_free_pages(GFP_KERNEL, 1);
	uhci->root_hub = hub;
	if (!hub)
		return NULL;

	uhci->fl = (struct uhci_framelist *) (PAGE_SIZE + (unsigned long) hub);

	/* Initialize the root hub */
	memset(hub, 0, sizeof(*hub));
	hub->uhci = uhci;
	hub->usb.maxchild = 2;
	usb_init_root_hub(&hub->usb);
	hub->usb.op = &uhci_root_operations;

	/*
	 * Initialize the queues. They all start out empty,
	 * linked to each other in the proper order.
	 */
	for (i = 0 ; i < 8; i++) {
		hub->qh[i].link = 2 | virt_to_bus(&hub->control_qh);
		hub->qh[i].element = 1;
	}
	
	hub->control_qh.link = 2 | virt_to_bus(&hub->bulk0_qh);
	hub->control_qh.element = 1;

	hub->bulk0_qh.link = 2 | virt_to_bus(&hub->bulk1_qh);
	hub->bulk0_qh.element = 1;

	hub->bulk1_qh.link = 2 | virt_to_bus(&hub->bulk2_qh);
	hub->bulk1_qh.element = 1;

	hub->bulk2_qh.link = 2 | virt_to_bus(&hub->bulk3_qh);
	hub->bulk2_qh.element = 1;

	hub->bulk3_qh.link = 1;
	hub->bulk3_qh.element = 1;

	/*
	 * Fill the frame list: make all entries point to
	 * the proper interrupt queue.
	 *
	 * This is probably silly, but it's a simple way to
	 * scatter the interrupt queues in a way that gives
	 * us a reasonable dynamic range for irq latencies.
	 */
	for (i = 0; i < 1024; i++) {
		struct uhci_qh * irq = &hub->interrupt2_qh;
		if (i & 1) {
			irq++;
			if (i & 2) {
				irq++;
				if (i & 4) { 
					irq++;
					if (i & 8) { 
						irq++;
						if (i & 16) {
							irq++;
							if (i & 32) {
								irq++;
								if (i & 64) {
									irq++;
								}
							}
						}
					}
				}
			}
		}
		uhci->fl->frame[i] =  2 | virt_to_bus(irq);
	}

	return uhci;
}


/*
 * De-allocate all resources..
 */
static void release_uhci(struct uhci *uhci)
{
	if (uhci->irq >= 0) {
		free_irq(uhci->irq, uhci);
		uhci->irq = -1;
	}

	if (uhci->root_hub) {
		free_pages((unsigned int) uhci->root_hub, 1);
		uhci->root_hub = NULL;
		uhci->fl = NULL;
	}
}

static int uhci_control_thread(void * __uhci)
{
	struct uhci *uhci = (struct uhci *)__uhci;

	lock_kernel();
	request_region(uhci->io_addr, 32, "usb-uhci");

	/*
	 * This thread doesn't need any user-level access,
	 * so get rid of all our resources..
	 */
	printk("uhci_control_thread at %p\n", &uhci_control_thread);
	exit_mm(current);
	exit_files(current);
	exit_fs(current);

	strcpy(current->comm, "uhci-control");

	/*
	 * Ok, all systems are go..
	 */
	start_hc(uhci);
	do {
		interruptible_sleep_on(&uhci_configure);
		uhci_check_configuration(uhci);
	} while (!signal_pending(current));

	reset_hc(uhci);
	release_region(uhci->io_addr, 32);

	release_uhci(uhci);
	MOD_DEC_USE_COUNT;
	return 0;
}	

/*
 * If we've successfully found a UHCI, now is the time to increment the
 * module usage count, start the control thread, and return success..
 */
static int found_uhci(int irq, unsigned int io_addr)
{
	int retval;
	struct uhci *uhci;

	uhci = alloc_uhci(io_addr);
	if (!uhci)
		return -ENOMEM;

	reset_hc(uhci);

	retval = -EBUSY;
	if (request_irq(irq, uhci_interrupt, SA_SHIRQ, "usb", uhci) == 0) {
		int pid;

		MOD_INC_USE_COUNT;
		uhci->irq = irq;
		pid = kernel_thread(uhci_control_thread, uhci, CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
		if (pid >= 0)
			return 0;

		MOD_DEC_USE_COUNT;
		retval = pid;
	}
	release_uhci(uhci);
	return retval;
}

static int init_uhci(struct pci_dev *dev)
{
	int i;

	/* Search for the IO base address.. */
	for (i = 0; i < 6; i++) {
		unsigned int io_addr = dev->base_address[i];

		/* IO address? */
		if (!(io_addr & 1))
			continue;

		io_addr &= PCI_BASE_ADDRESS_IO_MASK;

		/* Is it already in use? */
		if (check_region(io_addr, 32))
			break;

		return found_uhci(dev->irq, io_addr);
	}
	return -1;
}

int init_module(void)
{
	int retval;
	struct pci_dev *dev = NULL;
	u8 type;

	retval = -ENODEV;
	for (;;) {
		dev = pci_find_class(PCI_CLASS_SERIAL_USB<<8, dev);
		if (!dev)
			break;
		/* Is it UHCI */
		pci_read_config_byte(dev, PCI_CLASS_PROG, &type);
		if(type!=0)
			continue;
		/* Ok set it up */
		retval = init_uhci(dev);
		if (retval < 0)
			continue;

		init_mouse();
		return 0;
	}
	return retval;
}
